/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import intrinsic.Array;
import intrinsic.Function;
import intrinsic.mx.collections.ArrayCollection;
import intrinsic.mx.controls.DataGrid;
import intrinsic.mx.controls.listClasses.ListBase;
import intrinsic.mx.events.ListEvent;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.internal.flex.TableItemRendererClassFactory;
import org.eclipse.swt.events.*;

public class Table extends Composite {
	int columnCount;
	TableColumn [] columns;
	ArrayCollection items;
	boolean headerVisible;

public Table (Composite parent, int style) {
	super (parent, checkStyle (style));
}

static int checkStyle (int style) {
	/*
	* To be compatible with Windows, force the H_SCROLL
	* and V_SCROLL style bits.  On Windows, it is not
	* possible to create a table without scroll bars.
	*/
	style |= SWT.H_SCROLL | SWT.V_SCROLL;
	/* Flex is always FULL_SELECTION */
	style |= SWT.FULL_SELECTION;
	return checkBits (style, SWT.SINGLE, SWT.MULTI, 0, 0, 0, 0);
}

TableItem _getItem (int index) {
	TableItem item = (TableItem) items.getItemAt(index);
	if (item != null) return item;
	if ((style & SWT.VIRTUAL) == 0) return item;
//	return items [index] = new TableItem (this, SWT.NULL, -1, false);
	return null;
}

void createItem (TableColumn column, int index) {
	if (!(0 <= index && index <= columnCount)) error (SWT.ERROR_INVALID_RANGE);
	if (columnCount == columns.length) {
		TableColumn [] newColumns = new TableColumn [columns.length + 4];
		System.arraycopy (columns, 0, newColumns, 0, columns.length);
		columns = newColumns;
	}
	System.arraycopy (columns, index, columns, index + 1, columnCount++ - index);
	columns [index] = column;
	column.createWidget();
	Array array = ((DataGrid)object).columns; 
	array.splice(index, 0, column.column);
	((DataGrid)object).columns = array;
	if (columnCount == 1) {
		((DataGrid)object).showHeaders = headerVisible;
	}
}

void createItem (TableItem item, int index) {
	int itemCount = items.length;
	if (!(0 <= index && index <= itemCount)) error (SWT.ERROR_INVALID_RANGE);
	item.createWidget();
	items.addItemAt(item, index);
}

void createHandle () {
	DataGrid widget = new DataGrid();
	if ((style & SWT.MULTI) != 0) widget.allowMultipleSelection = true;
	widget.dataProvider = items = new ArrayCollection();
	widget.showHeaders = false;
	widget.itemRenderer = new TableItemRendererClassFactory(this, null, false);
	object = widget;
}

void createWidget () {
	super.createWidget ();
	columns = new TableColumn [4];
}

boolean checkData (TableItem item) {
	if (item.cached) return true;
	if ((style & SWT.VIRTUAL) != 0) {
		item.cached = true;
		Event event = new Event ();
		event.item = item;
		event.index = indexOf (item);
//		int mask = OS.G_SIGNAL_MATCH_DATA | OS.G_SIGNAL_MATCH_ID;
//		int signal_id = OS.g_signal_lookup (OS.row_changed, OS.gtk_tree_model_get_type ());
//		OS.g_signal_handlers_block_matched (modelHandle, mask, signal_id, 0, 0, 0, handle);
//		currentItem = item;
//		sendEvent (SWT.SetData, event);
//		//widget could be disposed at this point
//		currentItem = null;
//		if (isDisposed ()) return false;
//		OS.g_signal_handlers_unblock_matched (modelHandle, mask, signal_id, 0, 0, 0, handle);
		if (item.isDisposed ()) return false;
	}
	return true;
}

public void addSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener (listener);
	addListener (SWT.Selection,typedListener);
	addListener (SWT.DefaultSelection,typedListener);
}

void changeHandler(ListEvent event) {
	postEvent (SWT.Selection);
}

native Function changeHandlerFunction ()/*{
	return changeHandler__Lmx_events_ListEvent_2;
}*/;

protected void checkSubclass () {
}

public Point computeSize (int wHint, int hHint, boolean changed) {
	checkWidget ();
	return computeNativeSize (object, wHint, hHint, changed);
}

public void clear (int index) {
}

public void clear (int start, int end) {
}

public void clear (int[] indices) {
}

public void clearAll () {
}

public void deselect (int[] indices) {
}

public void deselect (int index) {
}

public void deselect (int start, int end) {
}

public void deselectAll () {
}

void destroyItem (TableItem item) {
	int index = items.getItemIndex(item);
	items.removeItemAt(index);
}

void destroyItem (TableColumn column) {
	Array array = ((DataGrid)object).columns;
	int index = array.indexOf(column.column, 0);
	array.splice(index, 1);
	((DataGrid)object).columns = array;
	System.arraycopy (columns, index + 1, columns, index, --columnCount - index);
	columns [columnCount] = null;
	if (columnCount == 0) {
		((DataGrid)object).showHeaders = false;
	}
}

public TableColumn getColumn (int index) {
	checkWidget();
	if (!(0 <= index && index < columnCount)) error (SWT.ERROR_INVALID_RANGE);
	return columns [index];
}

public int getColumnCount () {
	checkWidget();
	return columnCount;
}

public int[] getColumnOrder () {
	return null;
}

public TableColumn[] getColumns () {
	checkWidget();
	TableColumn [] result = new TableColumn [columnCount];
	System.arraycopy (columns, 0, result, 0, columnCount);
	return result;
}

public int getGridLineWidth () {
	return 0;
}

public int getHeaderHeight () {
	return (int)((DataGrid)object).headerHeight;
}

public boolean getHeaderVisible () {
	return headerVisible;
}

public TableItem getItem (int index) {
	checkWidget ();
	int itemCount = items.length;
	if (!(0 <= index && index < itemCount)) error (SWT.ERROR_INVALID_RANGE);
	return _getItem (index);
}

public TableItem getItem (Point point) {
	return null;
}

public int getItemCount () {
	return items.length;
}

public int getItemHeight () {
	return (int)((ListBase)object).rowHeight;
}

public TableItem[] getItems () {
	int itemCount = items.length;
	TableItem[] result = new TableItem[itemCount]; 
	for (int i=0; i<itemCount; i++) {
		result[i] = (TableItem)items.getItemAt(i);
	}
	return result;
}

public boolean getLinesVisible () {
	return true;
}

public TableItem[] getSelection () {
	checkWidget ();
	Array array = ((ListBase)object).selectedIndices;
	TableItem[] result = new TableItem[array.length];
	for (int i = 0; i < result.length; i++) {
		result[i] = (TableItem)items.getItemAt(arrayGetInt(array, i));
	}
	return result;
}

public int getSelectionCount () {
	return 0;
}

public int getSelectionIndex () {
	return 0;
}

public int[] getSelectionIndices () {
	return null;
}

public TableColumn getSortColumn () {
	return null;
}

public int getSortDirection () {
	return 0;
}

public int getTopIndex () {
	return 0;
}

public int indexOf (TableColumn column) {
	checkWidget ();
	if (column == null) error (SWT.ERROR_NULL_ARGUMENT);
	Array array = ((DataGrid)object).columns;
	return array.indexOf(column.column, 0);
}

public int indexOf (TableItem item) {
	checkWidget();
	return items.getItemIndex(item);
}

void invalidateItem (TableItem item, int columnIndex) {
	items.itemUpdated (item);
}

public boolean isSelected (int index) {
	return false;
}

void hookEvents (){
	super.hookEvents();
	object.addEventListener(ListEvent.CHANGE, changeHandlerFunction ());
}

void releaseChildren (boolean destroy) {
	if (items != null) {
		int itemCount = items.length;
		for (int i=0; i<itemCount; i++) {
			TableItem item = (TableItem)items.getItemAt(i);
			if (item != null && !item.isDisposed ()) {
				item.release (false);
			}
		}
	}
	if (columns != null) {
		for (int i=0; i<columnCount; i++) {
			TableColumn column = columns [i];
			if (column != null && !column.isDisposed ()) {
				column.release (false);
			}
		}
		columns = null;
	}
	super.releaseChildren (destroy);
}

void releaseWidget () {
	super.releaseWidget ();
	items = null;
}

public void remove (int[] indices) {
}

public void remove (int index) {
}

public void remove (int start, int end) {
}

public void removeAll () {
	checkWidget ();
	items.removeAll ();
}

public void removeSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);	
}

public void select (int[] indices) {
}

public void select (int index) {
}

public void select (int start, int end) {
}

public void selectAll () {
}

public void setColumnOrder (int[] order) {
}

public void setFont (Font font) {
}

public void setHeaderVisible (boolean show) {
	checkWidget ();
	headerVisible = show;
	if (columnCount > 0) {
		((DataGrid)object).showHeaders = show;
	}
}

public void setItemCount (int count) {
}

public void setLinesVisible (boolean show) {
}

public void setRedraw (boolean redraw) {
}

public void setSelection (int[] indices) {
}

public void setSelection (TableItem item) {
}

public void setSelection (TableItem[] items) {
}

public void setSelection (int index) {
}

public void setSelection (int start, int end) {
}

public void setSortColumn (TableColumn column) {
}

public void setSortDirection (int direction) {
}

public void setTopIndex (int index) {
}

public void showColumn (TableColumn column) {
}

public void showItem (TableItem item) {
}

public void showSelection () {
}

}
